<#
    PROJECT X Image Optimizer – Dependency Installer

    This PowerShell script prepares a fresh Windows machine to run the
    WebP converter GUI.

    What it does:
      1. Verifies that PowerShell is running with Administrator privileges.
      2. Checks for the Windows Package Manager (winget) and installs Python 3.12 LTS if missing.
      3. Ensures `pip` is available and upgrades it to the latest version.
      4. Installs or upgrades the Pillow imaging library.

    Usage:
      Right-click -> “Run with PowerShell” (recommended) or run from an
      elevated PowerShell window:

          Set-ExecutionPolicy -Scope Process -ExecutionPolicy Bypass
          .\install_dependencies.ps1

    The script is idempotent—runnning it again simply checks that the
    prerequisites remain installed and up-to-date.
#>

param(
    [string]$PythonPackageId = "Python.Python.3.12",
    [string]$PythonMinimumVersion = "3.10.0"
)

function Assert-Administrator {
    $currentIdentity = [Security.Principal.WindowsIdentity]::GetCurrent()
    $principal = New-Object Security.Principal.WindowsPrincipal($currentIdentity)
    if (-not $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
        Write-Warning "This script should run with Administrator privileges."
        Write-Host "Right-click PowerShell and choose 'Run as administrator', then rerun the script." -ForegroundColor Yellow
        throw "Administrator privileges required."
    }
}

function Assert-Winget {
    if (Get-Command winget -ErrorAction SilentlyContinue) {
        return
    }
    Write-Warning "winget (Windows Package Manager) is required but not detected."
    Write-Host "Install winget from the Microsoft Store or run the script again after updating Windows." -ForegroundColor Yellow
    throw "winget not available."
}

function Get-PythonPath {
    try {
        $python = Get-Command python -ErrorAction Stop
        return $python.Path
    } catch {
        return $null
    }
}

function Get-PythonVersion {
    $pythonPath = Get-PythonPath
    if (-not $pythonPath) {
        return $null
    }
    $version = & python --version 2>$null
    if (-not $version) {
        return $null
    }
    return ($version -split "\s+")[-1]
}

function Compare-Version {
    param(
        [Parameter(Mandatory)][string]$Current,
        [Parameter(Mandatory)][string]$Required
    )

    try {
        return ([System.Version]$Current) -ge ([System.Version]$Required)
    } catch {
        return $false
    }
}

function Install-Python {
    Write-Host "Installing Python via winget (this may take a moment)..." -ForegroundColor Cyan
    winget install --id $PythonPackageId --source winget --accept-package-agreements --accept-source-agreements -e
}

function Ensure-Python {
    $currentVersion = Get-PythonVersion
    if ($null -eq $currentVersion) {
        Write-Host "Python not detected; installing..." -ForegroundColor Cyan
        Install-Python
        $currentVersion = Get-PythonVersion
        if ($null -eq $currentVersion) {
            Write-Warning "Python was installed but is not yet available in the current PowerShell session. Close this window, reopen PowerShell as Administrator, and rerun the script if pip commands fail."
        }
        return
    }

    if (-not (Compare-Version -Current $currentVersion -Required $PythonMinimumVersion)) {
        Write-Host "Existing Python version $currentVersion is older than required ($PythonMinimumVersion). Updating..." -ForegroundColor Cyan
        Install-Python
        $updatedVersion = Get-PythonVersion
        if ($updatedVersion) {
            Write-Host "Python $updatedVersion is now installed." -ForegroundColor Green
        }
    } else {
        Write-Host "Python $currentVersion already installed." -ForegroundColor Green
    }
}

function Ensure-Pip {
    Write-Host "Ensuring pip is present and up to date..." -ForegroundColor Cyan
    & python -m ensurepip --upgrade
    & python -m pip install --upgrade pip
}

function Ensure-Pillow {
    Write-Host "Installing/upgrading Pillow..." -ForegroundColor Cyan
    & python -m pip install --upgrade pillow
}

try {
    Assert-Administrator
    Assert-Winget
    Ensure-Python
    Ensure-Pip
    Ensure-Pillow
    Write-Host "All dependencies are ready!" -ForegroundColor Green
} catch {
    Write-Error $_
    exit 1
}

exit 0

